<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';

// Check if user has any user management permissions
$userPermissions = [
    'manage_users', 'view_users', 'add_user', 'edit_user', 'delete_user'
];

if (!hasAnyPermission($userPermissions)) {
    $_SESSION['error'] = "You don't have permission to access user management.";
    header('Location: dashboard.php');
    exit;
}

$activePage = 'users';

$success = '';
$error = '';

// Handle user deletion
if (isset($_POST['delete_user']) && hasPermission('manage_users')) {
    try {
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ? AND id != ?");
        $stmt->execute([$_POST['delete_user'], $_SESSION['user_id']]);
        logSystemAction('user_deleted', "Deleted user ID: $_POST[delete_user]");
        $success = 'User deleted successfully!';
    } catch (PDOException $e) {
        $error = 'Error deleting user: ' . $e->getMessage();
    }
}

// Handle user creation/update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    try {
        if ($_POST['action'] === 'create') {
            // Validate required fields
            if (empty($_POST['username']) || empty($_POST['password']) || empty($_POST['name']) || empty($_POST['email']) || empty($_POST['role_id'])) {
                throw new Exception('All fields are required');
            }

            $stmt = $pdo->prepare("INSERT INTO users (username, password, full_name, email, role_id) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([
                $_POST['username'],
                password_hash($_POST['password'], PASSWORD_DEFAULT),
                $_POST['name'], // Changed from full_name to name to match form field
                $_POST['email'],
                $_POST['role_id']
            ]);
            $user_id = $pdo->lastInsertId();
            logSystemAction('user_created', "Created user: $_POST[username] (ID: $user_id)");
            $success = 'User created successfully!';
        } elseif ($_POST['action'] === 'update') {
            // Validate required fields
            if (empty($_POST['username']) || empty($_POST['name']) || empty($_POST['email']) || empty($_POST['role_id'])) {
                throw new Exception('All fields are required');
            }

            $sql = "UPDATE users SET username = ?, full_name = ?, email = ?, role_id = ?";
            $params = [$_POST['username'], $_POST['name'], $_POST['email'], $_POST['role_id']];
            
            if (!empty($_POST['password'])) {
                $sql .= ", password = ?";
                $params[] = password_hash($_POST['password'], PASSWORD_DEFAULT);
            }
            
            $sql .= " WHERE id = ?";
            $params[] = $_POST['user_id'];
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            logSystemAction('user_updated', "Updated user: $_POST[username] (ID: $_POST[user_id])");
            $success = 'User updated successfully!';
        }
    } catch (Exception $e) {
        $error = 'Error saving user: ' . $e->getMessage();
    }
}

// Handle enable/disable user actions
if (isset($_POST['disable_user']) && isset($_POST['user_id']) && hasPermission('manage_users')) {
    $stmt = $pdo->prepare("UPDATE users SET enabled = 0 WHERE id = ?");
    $stmt->execute([$_POST['user_id']]);
    $success = "User disabled successfully!";
}
if (isset($_POST['enable_user']) && isset($_POST['user_id']) && hasPermission('manage_users')) {
    $stmt = $pdo->prepare("UPDATE users SET enabled = 1 WHERE id = ?");
    $stmt->execute([$_POST['user_id']]);
    $success = "User enabled successfully!";
}

// Get all users with their roles
$stmt = $pdo->query("
    SELECT u.*, r.name as role_name 
    FROM users u 
    JOIN roles r ON u.role_id = r.id 
    ORDER BY u.username
");
$users = $stmt->fetchAll();

// Get all roles for the form
$stmt = $pdo->query("SELECT * FROM roles ORDER BY name");
$roles = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Users - EDDUH COMPUTERS</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { 
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); 
            font-family: 'Inter', sans-serif; 
            min-height: 100vh;
        }
        .card { 
            border-radius: 16px; 
            box-shadow: 0 4px 20px rgba(37,99,235,0.08); 
            border: none;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
        }
        .card-header { 
            background: none; 
            border-bottom: 1px solid rgba(37,99,235,0.1);
            padding: 1.5rem;
        }
        .btn {
            border-radius: 8px;
            font-weight: 600;
            padding: 0.5rem 1rem;
            transition: all 0.3s ease;
        }
        .btn-primary {
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            border: none;
            box-shadow: 0 2px 8px rgba(37,99,235,0.2);
        }
        .btn-primary:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(37,99,235,0.3);
        }
        .btn-warning {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
            border: none;
            box-shadow: 0 2px 8px rgba(245,158,11,0.2);
        }
        .btn-danger {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            border: none;
            box-shadow: 0 2px 8px rgba(239,68,68,0.2);
        }
        .table { 
            background: #fff; 
            border-radius: 12px; 
            overflow: hidden;
            box-shadow: 0 2px 12px rgba(0,0,0,0.04);
        }
        .table th { 
            background: #f8fafc;
            font-weight: 600;
            color: #1e293b;
            border-bottom: 2px solid #e2e8f0;
            padding: 1rem;
        }
        .table td { 
            padding: 1rem;
            vertical-align: middle;
            color: #475569;
        }
        .table-striped > tbody > tr:nth-of-type(odd) { 
            background-color: #f8fafc; 
        }
        .table-striped > tbody > tr:hover {
            background-color: #f1f5f9;
        }
        .modal-content { 
            border-radius: 16px;
            border: none;
            box-shadow: 0 4px 24px rgba(0,0,0,0.1);
        }
        .modal-header {
            border-bottom: 1px solid #e2e8f0;
            padding: 1.5rem;
        }
        .modal-body {
            padding: 1.5rem;
        }
        .form-label { 
            font-weight: 600; 
            color: #1e293b;
            margin-bottom: 0.5rem;
        }
        .form-control {
            border-radius: 8px;
            border: 1px solid #e2e8f0;
            padding: 0.75rem 1rem;
            transition: all 0.3s ease;
        }
        .form-control:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37,99,235,0.1);
        }
        .modal-title { 
            color: #1e293b; 
            font-weight: 700;
            font-size: 1.25rem;
        }
        .badge {
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            font-weight: 600;
        }
        .badge.bg-success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%) !important;
        }
        .badge.bg-danger {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%) !important;
        }
        .alert {
            border-radius: 12px;
            border: none;
            padding: 1rem 1.5rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        .alert-success {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            color: #065f46;
        }
        .alert-danger {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            color: #991b1b;
        }
        .section-header {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        .section-header i {
            color: #2563eb;
        }
        .btn-sm {
            padding: 0.4rem 0.75rem;
            font-size: 0.875rem;
        }
        .btn-group-sm > .btn {
            padding: 0.4rem 0.75rem;
            font-size: 0.875rem;
        }
        .table-responsive {
            border-radius: 12px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.04);
        }
        .avatar-circle {
            width: 32px;
            height: 32px;
            background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h4 class="section-header mb-0">
                            <i class="fas fa-users"></i>
                            User Management
                        </h4>
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#userModal" data-permission="add_user" data-permission-label="Add User">
                            <i class="fas fa-plus me-2"></i>Add New User
                        </button>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle me-2"></i>
                                <?php echo $success; ?>
                            </div>
                        <?php endif; ?>
                        <?php if ($error): ?>
                            <div class="alert alert-danger">
                                <i class="fas fa-exclamation-circle me-2"></i>
                                <?php echo $error; ?>
                            </div>
                        <?php endif; ?>

                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th><i class="fas fa-user me-2"></i>Username</th>
                                        <th><i class="fas fa-id-card me-2"></i>Name</th>
                                        <th><i class="fas fa-envelope me-2"></i>Email</th>
                                        <th><i class="fas fa-user-tag me-2"></i>Role</th>
                                        <th><i class="fas fa-toggle-on me-2"></i>Status</th>
                                        <th><i class="fas fa-calendar me-2"></i>Created</th>
                                        <th><i class="fas fa-cogs me-2"></i>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($users as $user): ?>
                                        <tr>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="avatar-circle me-2">
                                                        <?php echo strtoupper(substr($user['username'], 0, 1)); ?>
                                                    </div>
                                                    <?php echo htmlspecialchars($user['username']); ?>
                                                </div>
                                            </td>
                                            <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($user['email']); ?></td>
                                            <td>
                                                <span class="badge bg-primary" style="background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%) !important;">
                                                    <?php echo htmlspecialchars($user['role_name']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($user['enabled']): ?>
                                                    <span class="badge bg-success">
                                                        <i class="fas fa-check-circle me-1"></i>Enabled
                                                    </span>
                                                <?php else: ?>
                                                    <span class="badge bg-danger">
                                                        <i class="fas fa-times-circle me-1"></i>Disabled
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo date('M d, Y', strtotime($user['created_at'])); ?></td>
                                            <td>
                                                <div class="btn-group">
                                                    <button type="button" class="btn btn-sm btn-primary edit-user" 
                                                            data-user='<?php echo htmlspecialchars(json_encode($user)); ?>'
                                                            data-bs-toggle="modal" data-bs-target="#userModal"
                                                            data-permission="edit_user" data-permission-label="Edit User">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                                        <?php if ($user['enabled']): ?>
                                                            <form method="post" class="d-inline">
                                                                <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                                <button type="submit" name="disable_user" class="btn btn-sm btn-warning" onclick="return confirm('Are you sure you want to disable this user?')" data-permission="manage_users" data-permission-label="Manage Users">
                                                                    <i class="fas fa-ban"></i>
                                                                </button>
                                                            </form>
                                                        <?php else: ?>
                                                            <form method="post" class="d-inline">
                                                                <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                                <button type="submit" name="enable_user" class="btn btn-sm btn-success" onclick="return confirm('Are you sure you want to enable this user?')" data-permission="manage_users" data-permission-label="Manage Users">
                                                                    <i class="fas fa-check"></i>
                                                                </button>
                                                            </form>
                                                        <?php endif; ?>
                                                        <form method="post" class="d-inline">
                                                            <input type="hidden" name="delete_user" value="<?php echo $user['id']; ?>">
                                                            <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this user? This action cannot be undone.')" data-permission="delete_user" data-permission-label="Delete User">
                                                                <i class="fas fa-trash"></i>
                                                            </button>
                                                        </form>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- User Modal -->
    <div class="modal fade" id="userModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-user me-2"></i>
                        <span id="modalTitle">Add New User</span>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form method="post" id="userForm">
                        <input type="hidden" name="action" value="create">
                        <input type="hidden" name="user_id" value="">
                        
                        <div class="mb-3">
                            <label class="form-label">Username</label>
                            <input type="text" class="form-control" name="username" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Full Name</label>
                            <input type="text" class="form-control" name="name" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="email" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Password</label>
                            <input type="password" class="form-control" name="password" id="password">
                            <small class="text-muted" id="passwordHelp">Leave blank to keep current password when editing</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Role</label>
                            <select class="form-select" name="role_id" required>
                                <?php foreach ($roles as $role): ?>
                                    <option value="<?php echo $role['id']; ?>"><?php echo htmlspecialchars($role['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="text-end">
                            <button type="button" class="btn btn-light me-2" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" class="btn btn-primary" data-permission="add_user" data-permission-label="Add User">Save User</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Handle edit user button clicks
            document.querySelectorAll('.edit-user').forEach(button => {
                button.addEventListener('click', function() {
                    const user = JSON.parse(this.dataset.user);
                    const form = document.getElementById('userForm');
                    const modalTitle = document.getElementById('modalTitle');
                    
                    // Update form fields
                    form.action.value = 'update';
                    form.user_id.value = user.id;
                    form.username.value = user.username;
                    form.name.value = user.full_name;
                    form.email.value = user.email;
                    form.role_id.value = user.role_id;
                    form.password.value = '';
                    
                    // Update modal title
                    modalTitle.textContent = 'Edit User';
                    
                    // Show password help text
                    document.getElementById('passwordHelp').style.display = 'block';
                });
            });
            
            // Reset form when modal is closed
            document.getElementById('userModal').addEventListener('hidden.bs.modal', function() {
                const form = document.getElementById('userForm');
                form.reset();
                form.action.value = 'create';
                form.user_id.value = '';
                document.getElementById('modalTitle').textContent = 'Add New User';
                document.getElementById('passwordHelp').style.display = 'none';
            });
        });
    </script>
    
    <?php 
    // Add permission modal components
    echo getPermissionModalCSS();
    echo getPermissionModalHTML();
    echo getPermissionModalJS();
    ?>
    
    <?php echo getUniversalPermissionModalHTML(); ?>
    <?php echo getUniversalPermissionModalJS(); ?>
    
    <?php include '../includes/footer.php'; ?>
</body>
</html> 